function [alpha p gof] = plFitDeg(obj,varargin)
%PLFITDEG Fit the degree of this matrix to a power-law distribution
%   [alpha p gof] = PLFITDEG(obj, ... param, paramValue, ... ) fits the
%   degree of AdjacencyMatrix obj to a power-law distribution.  This
%   calls the function plfit of Clauset et. al. The return values are:
%       alpha - the alpha value for the power-law fit
%       p - p-value for this fit (according to the heurisitic in Clauset
%           et. al. a p-value less than .1 indicates the power law
%           has low probability of generating a data set with as poor a K-S
%           statistic as this data set.
%       gof - Kolmogorov goodness of fit statistic for this fit
%
%   Optional parameters:
%       'degreeType' - should be 'in', 'out', or 'total'.  Default is
%           'total'.  Determines the type of degree to be fit (in-degree,
%           out-degree, or total degree)
%       'weighted' - should be true or false.  Determines whether edge
%           weights are considered.  Default is true.
%       'tolerance' - if 'weighted' is set to false, then the
%           tolerance will determine the lower limit for the weight of an
%           edge. Default is 1.  Set to a higher number to eliminate noise
%           and spurious edges.
%       'pTest' - if set to 'false', this will skip the p-value
%           calculation.  This calculation is time-intensive, so set this
%           parameter to speed up the function.  Default is 'true'.

weighted = true;
degreeType = 'total';
tolerance = 1;
calcPVal = true;

for k = 1:2:length(varargin)
    switch varargin{k}
        case 'degreeType'
            degreeType = varargin{k+1};
        case 'tolerance'
            tolerance = varargin{k+1};
        case 'weighted'
            weighted = varargin{k+1};
        case 'pTest'
            calcPVal = varargin{k+1};
    end
end

switch weighted
    case true
        A = obj.A;
    case false
        A = obj.A;
        A = double(A>= tolerance);
end

switch degreeType
    case 'in'
        x = sum(A,1);
    case 'out'
        x = sum(A,2);
    case 'total'
        switch weighted
            case true
                A = getSquareMatrix(obj);
                colSums = sum(A,1);
                rowSums = sum(A,2);
                x = rowSums + colSums';
            case false
                A = getSquareMatrix(obj);
                A = double(A >= tolerance);
                colSums = sum(A,1);
                rowSums = sum(A,2);
                x = rowSums + colSums';
        end
end

[alpha,xmin,L] = plfit(x,'finite');

% plot the results
h = plplot(x,xmin,alpha);
if weighted, w = 'weighted'; else w = 'unweighted'; end
title(gca,sprintf('Power-law fit of %s %s degree',w,degreeType));
fprintf('Xmin: %.0f\n',xmin);
fprintf('alpha: %.5f\n',alpha);
p = 0;
gof = 0;
if calcPVal
    [p gof] = plpva(x,1,'silent');
    fprintf('p-value: %.5f\n',p);
    fprintf('Kolmogorov-Smirnov goodness of fit statistic: %.5f\n',gof);
end


end

